clear
set more 1
capture log close
set logtype text
set memory 300m

*
* Control program for the PSID extract.
*
* Written by Karen Dynan (August-October 2006) and modified by John Soroushian and
* Karen Dynan (2012)
*
* Each major section of this program is marked with a "*=========".
*
*============================================================================
* Choose what to run.
*
*   runind  = 1  ==> read individual files
*           = 0  ==> skip individual files
*
*   runfam  = 1  ==> read family files
*           = 0  ==> skip family files
*
*   stfam   = yyyy ==> first family file to read is yyyy
*   endfam  = yyyy ==> last family file to read is yyyy 
*
*   runwide = 1  ==> create "wide" datasets of each variable
*           = 0  ==> skip 
*
*   endwide = yyyy ==> last year of wide dataset is 19yy 
*
*   run11   = 1 ==> run the set of programs that incorporate the preliminary
*                   2011 data (once the full data come out, we can incorporate
*                   it into the above programs and drop each section
*
*   process = 1  ==> process the wide datasets to create new variables
*           = 0  ==> skip
*
*   endproc = yyyy ==> last year of processed wide dataset is yyyy

  local runind  = 0
  local runfam  = 0
  local stfam   = 1968
  local endfam  = 2009
  local runwide = 0
  local endwide = 2009
  local run11   = 0
  local process = 1

*============================================================================
* Read in the individual data.  The program readind reads in from a text
* data set and keeps the variables of interest (relationship to head and
* sequence numbers so we can identify the head in each year; the 1968 family
* and person numbers so we can create a unique identifier for the head;
* the household id numbers for each year which we'll need for the recursive
* merging).  Then, we loop through each year and create a data set with 
* the heads for that year, the unique head identifiers, and the family ids for
* that person in past years.

  if `runind' == 1 {

    set memory 300m
    log using logfiles/read_psid_ind, replace

      * This program creates a data set of relevant variables for all 
      * individuals.  It will form the basis of the family merge below.

        unzipfile psid_data/Individual/IND2009ER.zip 
        do readind
        erase IND2009ER.txt

      * Data sets of just the head each year.  When we do the merge,
      * we'll have one observation for anyone who has ever been a head
      * and we'll want to mark the years in which this person was the
      * head (the others years will be populated with the records for
      * the household in which that person resided even though he/she
      * wasn't the head).

      * 1968 is done separately, as the code for identifying the head is different.

        use "ind_datasets/inddat"
        keep if relat68 == 1
        gen headid68 = persid
        label variable headid68 "UNIQUE HEAD IDENTIFIER"

        keep headid68 id68
        sort id68
        summ
        compress
        save ind_datasets/ind68, replace

        use "ind_datasets/inddat"
        keep if relat68 == 2
        gen wifeid68 = persid
        label variable wifeid68 "UNIQUE IDENTIFIER (WIFE)"
        keep wifeid68 id68
        sort id68
        summ
        compress
        save ind_datasets/wife68, replace

      * Do the other years

        forvalues yyyy = 1969/2009 {

          if `yyyy'==1998 | `yyyy'==2000 | `yyyy'==2002 | `yyyy'==2004 | `yyyy'==2006 | `yyyy'==2008 {
            display "No PSID in `yyyy'"
            continue
          }    
          
          use "ind_datasets/inddat"

            * This bit of code parses out the last two digits of the year
            * since the datasets are labelled that way.

              scalar temptemp = string(`yyyy')
              local yy = substr(temptemp,3,2)
              disp "The last two digits for this loop are `yy'"
              save tempdat, replace
      
            * Keep just the heads.  The PSID manual explains why you need to use both the
            * the relat and seq variables.

              keep if (relat`yy'==1|relat`yy'==10) & seq`yy' < 50
        
            * Make the unique identifier.

              gen headid`yy' = persid
              label variable headid`yy' "UNIQUE HEAD IDENTIFIER"

            * Keep just the headid and the interview number.

              keep headid`yy' id`yy'
              compress
              summ
              sort id`yy'
              save ind_datasets/ind`yy', replace
              display "The file ind`yy' has been saved."
              clear

            * Now do the wives

              use tempdat
              keep if (relat`yy'==2|relat`yy'==20|relat`yy'==22) & seq`yy' < 50

            * Make the unique identifier.

              gen wifeid`yy' = persid
              label variable wifeid`yy' "UNIQUE IDENTIFIER (WIFE)"

            * Keep just the headid and the interview number.

              keep wifeid`yy' id`yy'
              compress
              summ
              sort id`yy'
              save ind_datasets/wife`yy', replace
              display "The file wife`yy' has been saved."

           clear

         }

    log close
    set memory 50m
  }

*============================================================================
* Read the family files.  

  if `runfam' == 1 {
    
    log using logfiles/read_psid_fam, replace

    forvalues yyyy = `stfam'/`endfam' {

      display " "
      display "Reading family data for `yyyy'"
      display " "
      if `yyyy'==1998 | `yyyy'==2000 | `yyyy'==2002 | `yyyy'==2004 | `yyyy'==2006 | `yyyy'==2008 {
        display "No PSID in `yyyy'"
        continue
      }    

      scalar temptemp = string(`yyyy')
      local yy = substr(temptemp,3,2)
	  
      * SAS extract
	  
        capture unzipfile psid_data/Family/`yy'FAM.zip      /* 1992 and earlier */
        capture unzipfile psid_data/Family/FAM`yyyy'.zip    /* 1993 */
		capture unzipfile psid_data/Family/FAM`yyyy'ER.zip  /* 1994 and later */
       
        saswrapper using readfam`yy'
        capture erase `yy'FAM.DAT
        capture erase FAM`yyyy'ER.txt
        capture erase FAM`yyyy'.txt
		
      do labren`yy'

      log close
      log using logfiles/read_psid`yy', replace

      ren id`yy'`yy' temptemp
      capture drop id*
      ren temptemp id`yy'

      * Merge in the unique headid identifier from the individual records

        sort id`yy'
        save tempdat, replace
        clear
        use ind_datasets/ind`yy'
        merge id`yy' using tempdat
        tab _merge
        drop _merge

      * Merge in the wife identifier from the individual records

        sort id`yy'
        save tempdat, replace
        clear
        use ind_datasets/wife`yy'
        merge id`yy' using tempdat
        tab _merge
        drop _merge


     * Clean up and save

        capture compress
	    sort id`yy'
        summ
        save yr_datasets/fam`yy', replace
        clear
        erase tempdat.dta
        log close
        log using logfiles/read_psid_fam, append

    }

  log close

  }
  

*============================================================================
* Create "wide" datasets for variables of interest.
*
*   Each data set will have variable for each year, and 'persid,' unique
*   person-level identifier to be used for merging.
*
*   Each observations corresponds to a person that has been, at some point,
*   a head of household.  The variable "head" can be used to identify whether
*   the person is a head in any given year.
*
*   For years in which the person is not a head, variables such as "age of 
*   head" provide information about the head for the household that the
*   person is in (the identify for the household can be obtained through
*   the "id" variables).
*

  if `runwide' == 1 {

    set memory 300m
	global mydate = subinstr("${S_DATE}"," ","-",.)
    capture log close
    quietly log using logfiles/makewide_$mydate, replace
    capture log close

    * makewide and makeid are the programs that do the heavy lifting

      capture program drop makewide
      run makewide
       
     * These are the variables for which we'll make wide data sets

  	   local vlist wgt_c
       local vlist `vlist' wgt_c_i     	   
	   local vlist `vlist' wgt_c_l     	   
       local vlist `vlist' hage     			   
       local vlist `vlist' wage     			
       local vlist `vlist' wifeid   			
       local vlist `vlist' hestat  	       
       local vlist `vlist' hestat_1			
       local vlist `vlist' westat  			    
       local vlist `vlist' westat_1  		    
       local vlist `vlist' acfdrs  			    
       local vlist `vlist' acfsmn 			      
       local vlist `vlist' acfsyr  			    
       local vlist `vlist' acfdhm  		  
       local vlist `vlist' fdhm_fs_acc   
       local vlist `vlist' fdhm_nfs_acc  
       local vlist `vlist' fddel_fs_acc  
       local vlist `vlist' fddel_nfs_acc 
       local vlist `vlist' fdrs_fs_acc   
       local vlist `vlist' fdrs_nfs_acc  
       local vlist `vlist' acrent   			    
       local vlist `vlist' achval   			
       local vlist `vlist' acutip  		     
       local vlist `vlist' hsex     			                  
       local vlist `vlist' race     			 
       local vlist `vlist' myp      			
       local vlist `vlist' hwtxyp   			    
       local vlist `vlist' hwtryp   			    
       local vlist `vlist' otxyp    			      
       local vlist `vlist' otryp    			
       local vlist `vlist' otaxp     		
       local vlist `vlist' hwtaxp   			
       local vlist `vlist' htlyp    			
       local vlist `vlist' wtlyp    			
       local vlist `vlist' hhryp    			
       local vlist `vlist' whryp    			
       local vlist `vlist' hillop   			                  
       local vlist `vlist' hillsp   			
       local vlist `vlist' willop   			                  
       local vlist `vlist' willsp   			
       local vlist `vlist' mtrp     			
       local vlist `vlist' fstd     			
       local vlist `vlist' intdat   			
       local vlist `vlist' intdat_m 			
       local vlist `vlist' hureas   			
       local vlist `vlist' wureas   			
       local vlist `vlist' fdhm        	    
       local vlist `vlist' fdrs        	    
       local vlist `vlist' fsmn        	   
       local vlist `vlist' fsyrp       	   
       local vlist `vlist' stincl      	
       local vlist `vlist' fdhm_fs_acc  	
       local vlist `vlist' fdhm_nfs_acc 	
       local vlist `vlist' fddel_fs_acc 	
       local vlist `vlist' fddel_nfs_acc	
       local vlist `vlist' fdrs_fs_acc  	
       local vlist `vlist' fdrs_nfs_acc 	
       local vlist `vlist' fdhm_fs_per  	
       local vlist `vlist' fdhm_nfs_per 	
       local vlist `vlist' fddel_fs_per 	
       local vlist `vlist' fddel_nfs_per	
       local vlist `vlist' fdrs_fs_per  	
       local vlist `vlist' fdrs_nfs_per 	
       local vlist `vlist' utilp       	
       local vlist `vlist' hsval       	                  
       local vlist `vlist' rent        	 
       local vlist `vlist' rent_amt     	 
       local vlist `vlist' rent_per      	 
       local vlist `vlist' nocars      	    
       local vlist `vlist' marit       	
       local vlist `vlist' famsz       	    
       local vlist `vlist' hsten       	    
       local vlist `vlist' heduc       	    
       local vlist `vlist' weduc       	 
       local vlist `vlist' hhsdeg      	 
       local vlist `vlist' whsdeg      	 
       local vlist `vlist' hcoll       	
       local vlist `vlist' wcoll       	 
       local vlist `vlist' hcolldeg    	 
       local vlist `vlist' wcolldeg    	
       local vlist `vlist' ssec        	
       local vlist `vlist' hssec       	
       local vlist `vlist' wssec       	
       local vlist `vlist' ossec       	
       local vlist `vlist' fstd        	
       local vlist `vlist' frmrec      	
       local vlist `vlist' ownbus      	
       local vlist `vlist' hbslyp      	
       local vlist `vlist' wbslyp      	       
       local vlist `vlist' veh_man_1     
       local vlist `vlist' veh_type_1     
       local vlist `vlist' veh_acq_1     
       local vlist `vlist' veh_new_1     
       local vlist `vlist' veh_yra_1     
       local vlist `vlist' veh_prc_1     
       local vlist `vlist' veh_dwn_1     
       local vlist `vlist' veh_bor_1     
       local vlist `vlist' veh_bora_1     
       local vlist `vlist' veh_pymt_1     
       local vlist `vlist' veh_pymtp_1     
       local vlist `vlist' veh_trm_1     
       local vlist `vlist' veh_pymtm_1     
       local vlist `vlist' veh_lse_amt_1     
       local vlist `vlist' veh_lse_per_1     
       local vlist `vlist' veh_man_2     
       local vlist `vlist' veh_type_2     
       local vlist `vlist' veh_acq_2     
       local vlist `vlist' veh_new_2     
       local vlist `vlist' veh_yra_2     
       local vlist `vlist' veh_prc_2     
       local vlist `vlist' veh_dwn_2     
       local vlist `vlist' veh_bor_2     
       local vlist `vlist' veh_bora_2     
       local vlist `vlist' veh_pymt_2     
       local vlist `vlist' veh_pymtp_2     
       local vlist `vlist' veh_trm_2     
       local vlist `vlist' veh_pymtm_2     
       local vlist `vlist' veh_lse_amt_2     
       local vlist `vlist' veh_lse_per_2     
       local vlist `vlist' veh_man_3     
       local vlist `vlist' veh_type_3     
       local vlist `vlist' veh_acq_3     
       local vlist `vlist' veh_new_3     
       local vlist `vlist' veh_yra_3     
       local vlist `vlist' veh_prc_3     
       local vlist `vlist' veh_dwn_3     
       local vlist `vlist' veh_bor_3     
       local vlist `vlist' veh_bora_3     
       local vlist `vlist' veh_pymt_3     
       local vlist `vlist' veh_pymtp_3     
       local vlist `vlist' veh_trm_3     
       local vlist `vlist' veh_pymtm_3     
       local vlist `vlist' veh_lse_amt_3     
       local vlist `vlist' veh_lse_per_3     
       local vlist `vlist' car_ins_amt     
       local vlist `vlist' car_ins_per     
       local vlist `vlist' add_lse_amt     
       local vlist `vlist' car_rep_amt     
       local vlist `vlist' gas_amt     
       local vlist `vlist' park_amt     
       local vlist `vlist' bus_amt     
       local vlist `vlist' cab_amt     
       local vlist `vlist' otrans_amt     
       local vlist `vlist' tuit_amt     
       local vlist `vlist' osch_amt     
       local vlist `vlist' hrep_amt     
       local vlist `vlist' hrep_per     
       local vlist `vlist' furn_amt     
       local vlist `vlist' furn_per     
       local vlist `vlist' cloth_amt     
       local vlist `vlist' cloth_per     
       local vlist `vlist' trip_amt     
       local vlist `vlist' trip_per     
       local vlist `vlist' orec_amt     
       local vlist `vlist' orec_per     
       local vlist `vlist' ptax_amt     
       local vlist `vlist' ptax_acc     
       local vlist `vlist' hins_amt     
       local vlist `vlist' mort_yes_1     
       local vlist `vlist' mort_typ_1     
       local vlist `vlist' mort_orig_1     
       local vlist `vlist' mort_bal_amt_1     
       local vlist `vlist' mort_bal_acc_1     
       local vlist `vlist' mort_pymt_1     
       local vlist `vlist' mort_pymt_acc_1     
       local vlist `vlist' mort_int_pp_1     
       local vlist `vlist' mort_int_frc_1     
       local vlist `vlist' mort_fxd_1     
       local vlist `vlist' mort_yr_1     
       local vlist `vlist' mort_trm_1     
       local vlist `vlist' mort_beh_yes_1     
       local vlist `vlist' mort_beh_mo_1     
       local vlist `vlist' mort_fc_yes_1     
       local vlist `vlist' mort_fc_mnth_1     
       local vlist `vlist' mort_fc_yr_1     
       local vlist `vlist' mort_mod_1     
       local vlist `vlist' mort_dist_prob_1     
       local vlist `vlist' mort_yes_2     
       local vlist `vlist' mort_typ_2     
       local vlist `vlist' mort_orig_2     
       local vlist `vlist' mort_bal_amt_2     
       local vlist `vlist' mort_bal_acc_2     
       local vlist `vlist' mort_pymt_2     
       local vlist `vlist' mort_pymt_acc_2     
       local vlist `vlist' mort_int_pp_2     
       local vlist `vlist' mort_int_frc_2     
       local vlist `vlist' mort_fxd_2     
       local vlist `vlist' mort_yr_2     
       local vlist `vlist' mort_trm_2     
       local vlist `vlist' mort_beh_yes_2     
       local vlist `vlist' mort_beh_mo_2     
       local vlist `vlist' mort_fc_yes_2     
       local vlist `vlist' mort_fc_mnth_2     
       local vlist `vlist' mort_fc_yr_2     
       local vlist `vlist' mort_mod_2     
       local vlist `vlist' mort_dist_prob_2     
       local vlist `vlist' fc_st_yes     
       local vlist `vlist' fc_st_mo     
       local vlist `vlist' fc_st_yr     
       local vlist `vlist' fc_end_yes     
       local vlist `vlist' fc_losthm     
       local vlist `vlist' fc_st_amt     
       local vlist `vlist' sechome_eqty     
       local vlist `vlist' veh_eqty     
       local vlist `vlist' bus_eqty     
       local vlist `vlist' stocks     
       local vlist `vlist' ira_yes     
       local vlist `vlist' ira_comp     
       local vlist `vlist' ira_amt     
       local vlist `vlist' chksav     
       local vlist `vlist' chksavira     
       local vlist `vlist' bonds     
       local vlist `vlist' oth_dbt     
       local vlist `vlist' adds_alts     
       local vlist `vlist' debt_out     
       local vlist `vlist' debt_in     
       local vlist `vlist' moved     
       local vlist `vlist' moved_mo     
       local vlist `vlist' moved_yr     
       local vlist `vlist' moved_why 
	   local vlist `vlist' state
	   local vlist `vlist' fsmn_per
	   local vlist `vlist' fsmn_amt
	   local vlist `vlist' fdrs_nfs_amt
	   local vlist `vlist' fdrs_fs_amt
	   local vlist `vlist' fddel_nfs_amt
       local vlist `vlist' fddel_fs_amt
	   local vlist `vlist' fdhm_nfs_amt
       local vlist `vlist' fdhm_fs_amt
	   local vlist `vlist' hgrade
	
          
     foreach vvv in `vlist' {
       makewide `vvv' 2009
       quietly log using logfiles/makewide_$mydate, append
       summ `vvv'*
       capture corr `1'*
       if _rc ~= 2000 & _rc~=2001 {
         corr `vvv'*    
        } 
       quietly log close
     }
 
    * Make a dataset with flags for whether person is the head

       use raw_w_datasets/temp_w
       keep head* persid
       drop headid*
       sort persid
       save raw_w_datasets/head_w, replace

     * Make a dataset with headid

       use raw_w_datasets/temp_w
       keep headid* persid
       sort persid
       save raw_w_datasets/headid_w, replace

     * Make a dataset with id numbers

       use raw_w_datasets/temp_w
       keep id* persid
       sort persid
       save raw_w_datasets/id_w, replace
		        
  set linesize 80
  	        
  }		        

* Run the preliminary 2011 files. We do it separately because these
* files have a very different format. When the real version comes out,
* we'll integrate it with the above.

  if `run11' == 1 {

    * SAS extract 
  
      unzipfile psid_data/Mortgage/foreclose.zip          
      *saswrapper using readmort11
      capture erase foreclose.sas7bdat
	  do labren11

	  log using logfiles/read_psid`yy', replace

    * note: id is still the 09 id so have to assume head/wife are the same. 
 
      rename id11 id09
      sort id09
      save tempdat, replace
      clear
      use ind_datasets\ind09.dta
      merge id09 using tempdat
      tab _merge
      drop _merge

      sort id09
      save tempdat, replace
      clear
      use ind_datasets\wife09.dta
      merge id09 using tempdat
      tab _merge
      drop _merge

      foreach vvv in id headid wifeid {
        rename `vvv'09 `vvv'11
      }
 
    * note:  we have too many people in here because we've used the 09 id numbers

      drop if hsval11 == .
      sort id11
      capture compress
      summ
      save yr_datasets/fam11, replace
      clear
      erase tempdat.dta
	
    log close

    set memory 300m
	global mydate = subinstr("${S_DATE}"," ","-",.)
    capture log close
    quietly log using logfiles/mw_11_$mydate, replace
    capture log close
    capture program drop makewide_11
    run makewide_11

    local vlist wifeid
    local vlist `vlist' acrent       
    local vlist `vlist' achval	
    local vlist `vlist' hsval   	                  
    local vlist `vlist' rent_amt  
    local vlist `vlist' rent_per  
    local vlist `vlist' hins_amt 
    local vlist `vlist' mort_yes_1      
    local vlist `vlist' mort_typ_1      
    local vlist `vlist' mort_orig_1     
    local vlist `vlist' mort_bal_amt_1  
    local vlist `vlist' mort_bal_acc_1  
    local vlist `vlist' mort_pymt_1     
    local vlist `vlist' mort_pymt_acc_1 
    local vlist `vlist' mort_int_pp_1   
    local vlist `vlist' mort_int_frc_1  
    local vlist `vlist' mort_fxd_1      
    local vlist `vlist' mort_yr_1       
    local vlist `vlist' mort_trm_1      
    local vlist `vlist' mort_beh_yes_1  
    local vlist `vlist' mort_beh_mo_1   
    local vlist `vlist' mort_fc_yes_1   
    local vlist `vlist' mort_fc_mnth_1  
    local vlist `vlist' mort_fc_yr_1    
    local vlist `vlist' mort_mod_1      
    local vlist `vlist' mort_dist_prob_1 
    local vlist `vlist' mort_yes_2       
    local vlist `vlist' mort_typ_2       
    local vlist `vlist' mort_orig_2      
    local vlist `vlist' mort_bal_amt_2   
    local vlist `vlist' mort_bal_acc_2   
    local vlist `vlist' mort_pymt_2      
    local vlist `vlist' mort_pymt_acc_2  
    local vlist `vlist' mort_int_pp_2    
    local vlist `vlist' mort_int_frc_2   
    local vlist `vlist' mort_fxd_2       
    local vlist `vlist' mort_yr_2        
    local vlist `vlist' mort_trm_2       
    local vlist `vlist' mort_beh_yes_2   
    local vlist `vlist' mort_beh_mo_2    
    local vlist `vlist' mort_fc_yes_2    
    local vlist `vlist' mort_fc_mnth_2   
    local vlist `vlist' mort_fc_yr_2     
    local vlist `vlist' mort_mod_2       
    local vlist `vlist' mort_dist_prob_2 
    local vlist `vlist' fc_st_yes        
    local vlist `vlist' fc_st_mo         
    local vlist `vlist' fc_st_yr         
    local vlist `vlist' fc_end_yes       
    local vlist `vlist' fc_losthm        
    local vlist `vlist' fc_st_amt        
    local vlist `vlist' sechome_eqty     
    local vlist `vlist' veh_eqty         
    local vlist `vlist' bus_eqty         
    local vlist `vlist' stocks           
    local vlist `vlist' ira_yes          
    local vlist `vlist' ira_comp         
    local vlist `vlist' ira_amt          
    local vlist `vlist' chksav           
    local vlist `vlist' bonds            
    local vlist `vlist' cc_dbt           
    local vlist `vlist' stud_dbt         
    local vlist `vlist' med_dbt          
    local vlist `vlist' leg_dbt          
    local vlist `vlist' rel_dbt         
    local vlist `vlist' adds_alts        
    local vlist `vlist' debt_out         
    local vlist `vlist' debt_in          
    local vlist `vlist' moved            
    local vlist `vlist' moved_mo         
    local vlist `vlist' moved_yr         
    local vlist `vlist' moved_why    
    local vlist `vlist' ownbus
 
    foreach vvv in `vlist' {
      makewide_11 `vvv' 2011
      quietly log using logfiles/mw_11_$mydate, append
      summ `vvv'*
      capture corr `1'*
      if _rc ~= 2000 & _rc~=2001 {
        corr `vvv'*    
      } 
      quietly log close
    }
 
* Make a wide dataset indicating who is a head
  use raw_w_datasets/temp_w
  keep head* persid
  drop headid*
  sort persid
  save raw_w_datasets/head_w, replace

* Make sure the 2011 data only have the families actually interviewed (have to do this
* because of complications arising from using the 2009 id's)

  use raw_w_datasets/id_w 
  capture drop id11
  gen id11 = id09
  sort persid
  merge persid using raw_w_datasets/hsval_w
  replace id11 = 0 if hsval11 == .
  drop hsval* _merge
  summ
  sort persid
  save, replace
  
  }
*============================================================================
* Run the program that processes the wide data sets.
*

  if `process' == 1 {

     do process

  }
